/* Copyright (c) 2005 Andrew Choi.  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright
   notice, this list of conditions and the following disclaimer in the
   documentation and/or other materials provided with the distribution.

3. All advertising materials mentioning features or use of this
   software must display the following acknowledgement:

     This product includes software developed by Andrew Choi.

4. The name "Andrew Choi" may not be used to endorse or promote
   products derived from this software without specific prior written
   permission.

THIS SOFTWARE IS PROVIDED BY ANDREW CHOI "AS IS" AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.  IN NO EVENT SHALL ANDREW CHOI BE LIABLE FOR ANY DIRECT,
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.  */

#include <unistd.h>

#include "SysexSendRequest.h"

bool InitializeSysexSendRequest(SysexSendRequest *request, MIDIEndpointRef endpoint)
{
  request->endpointRef = endpoint;
  request->finishedSending = true;   /* don't hang terminateSysexSeqSendReq!  */
  request->errorOccurred = false;    /* note that error may occur later in the sequence.  */
  
  return true;  /* always succeed.  */
}

void TerminateSysexSendRequest(SysexSendRequest *request)
{
  /* If an error has occurred, the completion proc call will never come.  */
  if (request->errorOccurred)
    return;
  
  /* This send request is truly completed only after the (possibly pending) last call to the completion proc has arrived. Only then can the SysexSendRequest data structure representing this send request be safely released by our caller.  */
  int count = 0;
  while (!request->finishedSending && count < 100) {
    usleep(100000);   /* test every 0.1 second.  */
    count++;	      /* wait at most 10 seconds.  */
  }
}

static void SendNextSysExMessage(SysexSendRequest *request);

static void CompletionProc(MIDISysexSendRequest *r)
{
  SysexSendRequest *request = (SysexSendRequest *)r->completionRefCon;
  
  if (!request->canceled && request->p < request->end && *request->p == 0xf0)
    SendNextSysExMessage(request);
  else {
    request->finishedSending = true;
    CFRelease(request->sysexDataToSend);
  }
}

static void SendNextSysExMessage(SysexSendRequest *request)
{
  const unsigned char *q = request->p + 1;
  while (q < request->end && *q != 0xf7)
    q++;
  if (q < request->end) {
    request->MIDISysexSendRequest.destination = request->endpointRef;
    request->MIDISysexSendRequest.data = (Byte *) request->p;
    request->MIDISysexSendRequest.bytesToSend = q - request->p + 1;
    request->MIDISysexSendRequest.complete = 0;
    request->MIDISysexSendRequest.completionProc = CompletionProc;
    request->MIDISysexSendRequest.completionRefCon = request;
    
    request->errorOccurred = (MIDISendSysex(&request->MIDISysexSendRequest) != noErr);
  }
  request->p = q + 1;
}

void SendSysexData(SysexSendRequest *request, CFDataRef sysexData)
{
  request->sysexDataToSend = CFRetain(sysexData);  /* retain the sysex data passed in to us in case caller disposes of it before send is complete.  */
  request->begin = (const unsigned char *) CFDataGetBytePtr(sysexData);
  request->p = request->begin;
  request->end = request->begin + CFDataGetLength(sysexData);
  request->canceled = false;
  request->finishedSending = false;
  
  if (request->p < request->end && *request->p == 0xf0)
    SendNextSysExMessage(request);
}

int NumOfBytesSent(SysexSendRequest *request)
{
  if (request->errorOccurred)
    return -1;
  
  return request->MIDISysexSendRequest.data - request->begin;
}

void CancelSend(SysexSendRequest *request)
{
  request->canceled = true;
  request->MIDISysexSendRequest.complete = 1;
}

